---
title: Amazon Bedrock · Cloudflare AI Gateway docs
description: Amazon Bedrock allows you to build and scale generative AI
  applications with foundation models.
lastUpdated: 2025-10-04T19:27:14.000Z
chatbotDeprioritize: false
source_url:
  html: https://developers.cloudflare.com/ai-gateway/usage/providers/bedrock/
  md: https://developers.cloudflare.com/ai-gateway/usage/providers/bedrock/index.md
---

[Amazon Bedrock](https://aws.amazon.com/bedrock/) allows you to build and scale generative AI applications with foundation models.

## Endpoint

```txt
https://gateway.ai.cloudflare.com/v1/{account_id}/{gateway_id}/aws-bedrock`
```

## Prerequisites

When making requests to Amazon Bedrock, ensure you have the following:

* Your AI Gateway Account ID.
* Your AI Gateway gateway name.
* An active Amazon Bedrock API token.
* The name of the Amazon Bedrock model you want to use.

## Make a request

When making requests to Amazon Bedrock, replace `https://bedrock-runtime.us-east-1.amazonaws.com/` in the URL you're currently using with `https://gateway.ai.cloudflare.com/v1/{account_id}/{gateway_id}/aws-bedrock/bedrock-runtime/us-east-1/`, then add the model you want to run at the end of the URL.

With Bedrock, you will need to sign the URL before you make requests to AI Gateway. You can try using the [`aws4fetch`](https://github.com/mhart/aws4fetch) SDK.

## Examples

### Use `aws4fetch` SDK with TypeScript

```typescript
import { AwsClient } from "aws4fetch";


interface Env {
  accessKey: string;
  secretAccessKey: string;
}


export default {
  async fetch(
    request: Request,
    env: Env,
    ctx: ExecutionContext,
  ): Promise<Response> {
    // replace with your configuration
    const cfAccountId = "{account_id}";
    const gatewayName = "{gateway_id}";
    const region = "us-east-1";


    // added as secrets (https://developers.cloudflare.com/workers/configuration/secrets/)
    const accessKey = env.accessKey;
    const secretKey = env.secretAccessKey;


    const awsClient = new AwsClient({
      accessKeyId: accessKey,
      secretAccessKey: secretKey,
      region: region,
      service: "bedrock",
    });


    const requestBodyString = JSON.stringify({
      inputText: "What does ethereal mean?",
    });


    const stockUrl = new URL(
      `https://bedrock-runtime.${region}.amazonaws.com/model/amazon.titan-embed-text-v1/invoke`,
    );


    const headers = {
      "Content-Type": "application/json",
    };


    // sign the original request
    const presignedRequest = await awsClient.sign(stockUrl.toString(), {
      method: "POST",
      headers: headers,
      body: requestBodyString,
    });


    // Gateway Url
    const gatewayUrl = new URL(
      `https://gateway.ai.cloudflare.com/v1/${cfAccountId}/${gatewayName}/aws-bedrock/bedrock-runtime/${region}/model/amazon.titan-embed-text-v1/invoke`,
    );


    // make the request through the gateway url
    const response = await fetch(gatewayUrl, {
      method: "POST",
      headers: presignedRequest.headers,
      body: requestBodyString,
    });


    if (
      response.ok &&
      response.headers.get("content-type")?.includes("application/json")
    ) {
      const data = await response.json();
      return new Response(JSON.stringify(data));
    }


    return new Response("Invalid response", { status: 500 });
  },
};
```
